/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/*-------------------------------------------------------------------------
 * FILE: timcheck.c
 * Author: Christoph Schmidt, BID R&D
 * This file contains the functions for the timing checker.
 *-------------------------------------------------------------------------*/

#include <typedefs.h>

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <regconst.h>
#include <session.h>
#include <regx3c.h>
#include <regx3d.h>
#include <regx3e.h>
#include <regx3f.h>
#include <board.h>
#include <timcheck.h>

static b_errtype err;

/* ------------------------------------------------------------------------
 * This function sets a mask for a signal
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckMaskSet(
    b_handletype handle,
    b_signaltype signal,
    b_int32 value
)
{
  b_int32 SigPosition,SigLength;
  b_int8 inbuf[6];
  b_int8 zw;
  
  err=B_E_OK;

  /* license and handle checking */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check if signal is allowed and value is in range */
  B_ERRCHECK(BestParamCheck(handle, B_PARAM_TCSIG, (b_int32) signal, value));
  B_ERRCHECK(BestTimCheckBitPosGet(handle,signal,&SigPosition,&SigLength));
   

  zw = (b_int8) SigPosition;
  (void) BestByteCopy(inbuf + 0, &zw, 1UL);
  zw = (b_int8) SigLength;
  (void) BestByteCopy(inbuf + 1, &zw, 1UL);
  (void) BestLong2Stream(inbuf + 2, &value, 1UL);
     
  B_ERRETURN(BestBasicCommand(handle, CMD_TCMASK_SET,inbuf, IN_TCMASK_SET, NULL, NULL)); 
}
/* ------------------------------------------------------------------------
 * This function sets a mask at a bitposition
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckBitMaskSet(
    b_handletype handle,
    b_int32 bitposition,
    b_int32 value
)
   /* this function is for internal purposes only */
{
  b_int8 inbuf[6];
  b_int8 zw;
  b_signaltype signal;
  
  err=B_E_OK;

  /* license and handle checking */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check if signal is allowed and value is in range */
  B_ERRCHECK(BestTimCheckSignalGet(handle,bitposition,&signal,NULL,NULL,NULL,NULL));
  B_ERRCHECK(BestParamCheck(handle, B_PARAM_TCSIG, (b_int32) signal, value));
   
  zw = (b_int8) bitposition;
  (void) BestByteCopy(inbuf + 0, &zw, 1UL);
  zw = (b_int8) 1; /* i.e. the length: only binary here; e.g. AD32[4] is binary, too */
  (void) BestByteCopy(inbuf + 1, &zw, 1UL);
  (void) BestLong2Stream(inbuf + 2, &value, 1UL);
     
  B_ERRETURN(BestBasicCommand(handle, CMD_TCMASK_SET,inbuf, IN_TCMASK_SET, NULL, NULL)); 
}

/* ------------------------------------------------------------------------
 * This function gets a mask for a signal
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckMaskGet(
    b_handletype handle,
    b_signaltype signal,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestTimCheckMaskGet [tcmget]");

  b_int32 SigPosition,SigLength;
  b_int8 inbuf[2],outbuf[4];
  b_int8 zw;
  b_param_infotype *ParamInfo;
  b_int16 outsize;

  err=B_E_OK;

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check if signal is allowed */
  if ((err = BestParamInfoGet(handle, B_PARAM_TCSIG,
	(b_int32)signal, &ParamInfo, (b_int32)B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_TCSIG,
	(b_int32)signal, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  B_ERRCHECK(BestTimCheckBitPosGet(handle,signal,&SigPosition,&SigLength));
    
  zw = (b_int8) SigPosition;
  (void) BestByteCopy(inbuf + 0, &zw, 1UL);
  zw = (b_int8) SigLength;
  (void) BestByteCopy(inbuf + 1, &zw, 1UL);
  outsize = OUT_TCMASK_GET;

  B_ERRCHECK(BestBasicCommand(handle, CMD_TCMASK_GET, inbuf,
  IN_TCMASK_GET, outbuf, &outsize));

  assert(outsize == OUT_TCMASK_GET);

  if (value)
  {
    (void) BestStream2Long(value,outbuf,1UL);
  }
  
  B_ERRETURN(B_E_OK);
}

/* ------------------------------------------------------------------------
 * This function gets a mask for a bitposition
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckBitMaskGet(
    b_handletype handle,
    b_int32 bitposition,
    b_int32 * value
)
{
  /* this function is for internal purposes only */
  
  B_DECLARE_FUNCNAME("BestTimCheckBitMaskGet [tcbmget]");

  b_int8 inbuf[2],outbuf[4];
  b_int8 zw;
  b_param_infotype *ParamInfo;
  b_int16 outsize;
  b_signaltype signal;

  err=B_E_OK;

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check if signal is allowed and value is in range */
  B_ERRCHECK(BestTimCheckSignalGet(handle,bitposition,&signal,NULL,NULL,NULL,NULL));

  /* Check if signal is allowed */
  if ((err = BestParamInfoGet(handle, B_PARAM_TCSIG,
	(b_int32)signal, &ParamInfo, (b_int32)B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_TCSIG,
	(b_int32)signal, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }
    
  zw = (b_int8) bitposition;
  (void) BestByteCopy(inbuf + 0, &zw, 1UL);
  zw = (b_int8) 1; /* signal length; only binary here (e.g. AD32[4] is binary, too */
  (void) BestByteCopy(inbuf + 1, &zw, 1UL);
  outsize = OUT_TCMASK_GET;

  B_ERRCHECK(BestBasicCommand(handle, CMD_TCMASK_GET, inbuf,
  IN_TCMASK_GET, outbuf, &outsize));

  assert(outsize == OUT_TCMASK_GET);

  if (value)
  {
    (void) BestStream2Long(value,outbuf,1UL);
  }
  
  B_ERRETURN(B_E_OK);
}

/* ------------------------------------------------------------------------
 * This function defaults the timing checker
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckDefaultSet(
    b_handletype handle
)
{
  b_generic_infotype *GenInfo;
  b_param_infotype *ParamInfo;
  b_int32 BusSpeed,i;
  
  /* license and handle checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Set mask to default */

  /* Get pointer to generic info */
  B_ERRCHECK(BestGenInfoGet(handle, B_PARAM_TCSIG, &GenInfo));

  /* Set default masks for all TC signals */
  for (i = 0; i < GenInfo->num_elem; i++)
  {
    /* Get pointer to i-th (existing) signal */
    B_ERRCHECK(BestParamInfoGet(handle, B_PARAM_TCSIG,
	i, &ParamInfo, (b_int32)B_INDEX_SEARCH));

    /* Set it to default */
    B_ERRCHECK(BestTimCheckMaskSet(handle,
        ParamInfo->proptyp.signalprop,
        ParamInfo->defaultval));
  }

  /* Set properties to default */
  /* Get pointer to generic info */
  B_ERRCHECK(BestGenInfoGet(handle, B_PARAM_TCPROP, &GenInfo));

  /* Set default vals for all TC props */
  for (i = 0; i < GenInfo->num_elem; i++)
  {
    /* Get pointer to i-th (existing) property */
    B_ERRCHECK(BestParamInfoGet(handle, B_PARAM_TCPROP,
	i, &ParamInfo, (b_int32)B_INDEX_SEARCH));

    if (ParamInfo->proptyp.tcprop==B_TC_SETUP_TIME)
    {
      /* Special case:
         Setup-time is dependent on the current frequency */
      B_ERRCHECK(BestSystemInfoGet(handle,B_SINFO_BUSSPEED,&BusSpeed));

      /* Set it to default */
      B_ERRCHECK(BestTimCheckPropSet(handle,
          ParamInfo->proptyp.tcprop,
          (BusSpeed>=33100000?3000:7000))); /* PCI-spec values */
    }
    else
    {
      /* Set property to default according to dynamic capabilities */
      B_ERRCHECK(BestTimCheckPropSet(handle,
          ParamInfo->proptyp.tcprop,
          ParamInfo->defaultval));
    }
  }
  
  /* Set generic property to default */
  B_ERRCHECK(BestTimCheckGenPropSet(handle,B_TCGEN_SPEC,1));
  
  B_ERRETURN (B_E_OK);
}


/* ------------------------------------------------------------------------
 * This function gets the timing checker status
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckStatusGet(
    b_handletype handle,
    b_tcstatustype tcstatus,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestTimCheckStatusGet [tcsget]");

  b_int8 inbuf;
  b_int8 outbuf[4];
  b_int16 outsize=OUT_TCSTATUS_READ;

  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check, wether tcstatus is a valid status at all */
  if ((err = BestParamInfoGet(handle, B_PARAM_TCSTATUS,
	(b_int32)tcstatus, NULL, (b_int32)B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_TCSTATUS,
	(b_int32)tcstatus, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  inbuf=(b_int8) tcstatus;
  B_ERRCHECK(BestBasicCommand(handle, CMD_TCSTATUS_READ,
    &inbuf, IN_TCSTATUS_READ, outbuf, &outsize));

  if (value)
  {
    (void) BestStream2Long(value, outbuf, 1UL);
  }
  
  B_ERRETURN(B_E_OK);

}

/* ------------------------------------------------------------------------
 * This function clears the accu-registers
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckStatusClear(
    b_handletype handle
)
{
  B_DECLARE_FUNCNAME("BestTimCheckStatusClear [tcsclear]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  B_ERRETURN(BestBasicCommand(handle, CMD_TCSTATUS_CLEAR, NULL, 0, NULL, NULL));
}

/* ------------------------------------------------------------------------
 * This function sets  properties
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTimCheckPropSet(  
    b_handletype handle,
    b_tcproptype tcprop,                    
    b_int32      value        
    )
{
  b_int8 zw;
  b_int8 inbuf [5];

  B_DECLARE_FUNCNAME("BestTimCheckPropSet [tcprpset]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);
  B_ERRCHECK(BestParamCheck(handle, B_PARAM_TCPROP, (b_int32) tcprop, value));

  zw = (b_int8) tcprop;
  (void) BestByteCopy(inbuf+0, &zw, 1UL);
  (void) BestLong2Stream(inbuf+1, &value, 1UL);
     
  B_ERRETURN(BestBasicCommand(handle, CMD_TCPROP_SET,inbuf, IN_TCPROP_SET, NULL, NULL)); 
}

/* ------------------------------------------------------------------------
 * This function gets properties
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckPropGet(  
    b_handletype handle,
    b_tcproptype tcprop,                    
    b_int32      *value        
    )
{
  b_int8 zw;
  b_int8 inbuf;
  b_int8 outbuf [4];
  b_param_infotype *ParamInfo;
  b_int16 outsize;

  B_DECLARE_FUNCNAME("BestTimCheckPropGet [tcprpget]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check if tcprop is allowed */
  if ((err = BestParamInfoGet(handle, B_PARAM_TCPROP,
	(b_int32)tcprop, &ParamInfo, (b_int32)B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_TCPROP,
	(b_int32)tcprop, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  zw=(b_int8) tcprop;
  (void) BestByteCopy(&inbuf, &zw, 1UL);
  outsize = OUT_TCPROP_GET;

  B_ERRCHECK(BestBasicCommand(handle, CMD_TCPROP_GET, &inbuf,
  IN_TCPROP_GET, outbuf, &outsize));

  if (value)
  {
    (void) BestStream2Long(value,outbuf,1UL);
  }
  
  B_ERRETURN(B_E_OK);

}

/* ------------------------------------------------------------------------
 * This function sets generic properties
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTimCheckGenPropSet(  
    b_handletype handle,
    b_tcgenproptype tcprop,                    
    b_int32      value        
    )
{
  b_int8 zw;
  b_int8 inbuf [5];

  B_DECLARE_FUNCNAME("BestTimCheckGenPropSet [tcgprpset]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);
  B_ERRCHECK(BestParamCheck(handle, B_PARAM_TCGENPROP, (b_int32) tcprop, value));

  zw = (b_int8) tcprop;
  (void) BestByteCopy(inbuf+0, &zw, 1UL);
  (void) BestLong2Stream(inbuf+1, &value, 1UL);
     
  B_ERRETURN(BestBasicCommand(handle, CMD_TCGENPROP_SET,inbuf, IN_TCGENPROP_SET, NULL, NULL)); 
}

/* ------------------------------------------------------------------------
 * This function gets generic properties
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckGenPropGet(  
    b_handletype handle,
    b_tcgenproptype tcprop,                    
    b_int32      *value        
    )
{
  b_int8 zw;
  b_int8 inbuf;
  b_int8 outbuf [4];
  b_param_infotype *ParamInfo;
  b_int16 outsize;

  B_DECLARE_FUNCNAME("BestTimCheckGenPropGet [tcgprpget]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Check if tcprop is allowed */
  if ((err = BestParamInfoGet(handle, B_PARAM_TCGENPROP,
	(b_int32)tcprop, &ParamInfo, (b_int32)B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_TCGENPROP,
	(b_int32)tcprop, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  zw=(b_int8) tcprop;
  (void) BestByteCopy(&inbuf, &zw, 1UL);
  outsize = OUT_TCGENPROP_GET;

  B_ERRCHECK(BestBasicCommand(handle, CMD_TCGENPROP_GET, &inbuf,
  IN_TCGENPROP_GET, outbuf, &outsize));

  if (value)
  {
    (void) BestStream2Long(value,outbuf,1UL);
  }
  
  B_ERRETURN(B_E_OK);

}
/* ------------------------------------------------------------------------
 * This function programs preparation register to HW
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckProg(
    b_handletype handle
    )
{    
  B_DECLARE_FUNCNAME("BestTimCheckProg [tcprog]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  B_ERRETURN(BestBasicCommand(handle, CMD_TCPROP_PROG, NULL, 0, NULL, NULL));
}

b_errtype EXPORT BestTimCheckRead(
    b_handletype          handle
    )
{
  B_DECLARE_FUNCNAME("BestTimCheckRead [tcread]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  B_ERRETURN(BestBasicCommand(handle, CMD_TCPROP_READ, NULL, 0, NULL, NULL));
}

/* ------------------------------------------------------------------------
 * This function returns the bitposition and signallength of a given signal
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckBitPosGet(
    b_handletype handle,
    b_signaltype signal,  
    b_int32 *position,
    b_int32 *length
)
{
  b_param_infotype *parinfo;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  *position = 0;
  *length = 0;

  err = BestParamInfoGet(handle, B_PARAM_TCSIG, (b_int32) signal, &parinfo,
    (b_int32) B_ENUM_SEARCH);

  if (err != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32) B_PARAM_TCSIG,
      (b_int32) signal, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  *position = parinfo->min_val; /* we found the signal and here is pos */
  *length = parinfo->max_val;   /* and here the width */

  B_ERRETURN(B_E_OK);
}

/* ------------------------------------------------------------------------
 * This function returns the id'th bit position
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckIndexBitPosGet (
  b_handletype	handle,
  b_int32 index,                  
  b_int32 *bitpos)
{
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  /* Currently we have 99 bitpositions:
     accu1: 0,..,31
     accu2:32,..,63
     accu3:64,..,95
     accu4:112,113,114
  */ 

  err=B_E_OK;

  if (index<96)
  {
    /* accu1,..,3 */
    if (bitpos) *bitpos=index;
  }
  else if (index<99)
  {
    /* accu4: only bits 16,17,18 (bitpos 112,113,114) used */
    if (bitpos) *bitpos=index+16;
  }
  else
  {
    /* no bitposition for this id */
    err=B_E_PARAM;
  }

  B_ERRETURN(err);
}



/* ------------------------------------------------------------------------
 * This function returns the signal for a given bit position
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTimCheckSignalGet(
    b_handletype handle,
    b_int32 bitposition,         /* e.g. 35  (i.e. AD32[3] */
    b_signaltype * signal,       /* B_SIG_AD32             */
    b_charptrtype *signalname,   /* "AD32"                 */
    b_int32 *offset,             /* 3  (from AD32[0] away) */
    b_int32 *signallength,       /* 32  (length of AD32)   */
    b_tcstatustype *accu         /* B_SIG_ACCU2            */
)
{
  B_DECLARE_FUNCNAME("BestTimCheckSignalGet");

  b_errtype errt;
  b_int32 i;
  b_generic_infotype *geninfo;  /* hold the generic infotype for timing checker signals */
                               
  b_param_infotype *parinfo;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  errt = B_E_INVALID_TC_SIG;  /* this is the default */

  /* get the generic information about timing checker signals */
  err = BestGenInfoGet(handle, B_PARAM_TCSIG, &geninfo);
  B_ERRCHECK(err);

  /* run through all signals and search for the correct bitposition */
  for (i = 0; i < geninfo->num_elem; i++)
  {
    err = BestParamInfoGet(handle, B_PARAM_TCSIG, i, &parinfo,
	(b_int32)B_INDEX_SEARCH);
    if (err)
      BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_TCSIG, i, 0UL, 0UL, 0UL);
    B_ERRCHECK(err);

    if (bitposition >= parinfo->min_val 
        && bitposition < parinfo->min_val+parinfo->max_val)  
    {
      if (signal) *signal = parinfo->proptyp.signalprop; 
      if (offset) *offset= bitposition-parinfo->min_val;
      if (signalname) *signalname=parinfo->param_name;
      if (signallength) *signallength=parinfo->max_val;
      if (accu)
      {
        if (bitposition<32)
	{
	  *accu=B_TC_ACCUSIG1;
	}
	else if (bitposition<64)
	{
	  *accu=B_TC_ACCUSIG2;
	}
	else if (bitposition<96)
	{
	  *accu=B_TC_ACCUSIG3;
	}
	else 
	{
          *accu=B_TC_ACCUSIG4;
	}
      }
      errt = B_E_OK;
      break;
    }
  }

  B_ERRETURN(errt);
}


/* ------------------------------------------------------------------------
 * This function returns a textual report
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTimCheckResultGet(
  b_handletype	handle,
  b_ccharptrtype *errortext
)
{
  b_int32 accu1,accu2,accu3,accu4,status;
  B_DECLARE_FUNCNAME("BestTimCheckResultGet[tcresultget]");
 
  B_FCT_PARAM_NULL_POINTER_CHK(errortext);
  
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  
  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);

  err=BestTimCheckStatusGet(handle,B_TC_ACCUSIG1,&accu1);B_ERRCHECK(err);
  err=BestTimCheckStatusGet(handle,B_TC_ACCUSIG2,&accu2);B_ERRCHECK(err);
  err=BestTimCheckStatusGet(handle,B_TC_ACCUSIG3,&accu3);B_ERRCHECK(err);
  err=BestTimCheckStatusGet(handle,B_TC_ACCUSIG4,&accu4);B_ERRCHECK(err);
  err=BestTimCheckStatusGet(handle,B_TC_TCSTAT,&status);B_ERRCHECK(err);

  err=BestTimCheckErrResultGet(handle,accu1,accu2,accu3,accu4,status,errortext);
  
  B_ERRETURN(err);
}

/* ------------------------------------------------------------------------
 * This function returns a textual report
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTimCheckErrResultGet(
  b_handletype	handle,
  b_int32 accu1,
  b_int32 accu2,
  b_int32 accu3,
  b_int32 accu4,
  b_int32 status,
  b_ccharptrtype *errortext
  )
{
  static char tc_err_result_buffer[0x1000];
  b_int32 i,bitoffset,len;
  b_generic_infotype *geninfo;  /* hold the generic infotype for timing checker signals */
  char buf[3]; 
  b_int32 violation=0; /* wether any violation has been found */
  b_charptrtype signame;

  B_DECLARE_FUNCNAME("BestTimCheckErrResultGet[tceresultget]");

  status; /* not used */

  if (!Best16BitRegisterFile(handle)) 
    B_ERRETURN(B_E_NOT_E2925A);

  B_FCT_PARAM_NULL_POINTER_CHK(errortext);
  
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  
  tc_err_result_buffer[0]='\0';
  strcat(tc_err_result_buffer,"\n**** TIMING CHECKER VIOLATIONS ****\n");        

  /* get the generic information about timing checker signals */
  err = BestGenInfoGet(handle, B_PARAM_TCSIG, &geninfo);
  B_ERRCHECK(err);

  if (accu1)
  {
    /* TC-violation somewhere in accu1 */
    for (i=0;i<32;i++)
    {
      /* check every bitposition */
      if (accu1&1<<i)
      {
        /* TC violation at bitposition i */
	violation=1;
        err=BestTimCheckSignalGet(handle,i,NULL,&signame,&bitoffset,&len,NULL);
	B_ERRCHECK(err);
	strcat(tc_err_result_buffer,signame);        
	if (len>1) 
	{ 
          /* Vector signal, e.g. AD32 */
	  strcat(tc_err_result_buffer,"[");        
  	  sprintf(buf,"%lu",bitoffset);
	  strcat(tc_err_result_buffer,buf);        
	  strcat(tc_err_result_buffer,"]");
        }
	strcat(tc_err_result_buffer," violated\n");
      }
    }
  }

  if (accu2)
  {
    /* TC-violation somewhere in accu2 */
    for (i=0;i<32;i++)
    {
      /* check every bitposition */
      if (accu2&1<<i)
      {
        /* TC violation at bitposition i+32 */
	violation=1;
        err=BestTimCheckSignalGet(handle,i+32,NULL,&signame,&bitoffset,&len,NULL);
	B_ERRCHECK(err);
	strcat(tc_err_result_buffer,signame);        
	if (len>1) 
	{ 
          /* Vector signal, e.g. AD32 */
	  strcat(tc_err_result_buffer,"[");        
  	  sprintf(buf,"%lu",bitoffset);
	  strcat(tc_err_result_buffer,buf);        
	  strcat(tc_err_result_buffer,"]");
        }
	strcat(tc_err_result_buffer," violated\n");
      }
    }
  }
	        
  if (accu3)
  {
    /* TC-violation somewhere in accu3 */
    for (i=0;i<32;i++)
    {
      /* check every bitposition */
      if (accu3&1<<i)
      {
        /* TC violation at bitposition i+64 */
	violation=1;
        err=BestTimCheckSignalGet(handle,i+64,NULL,&signame,&bitoffset,&len,NULL);
	B_ERRCHECK(err);
	strcat(tc_err_result_buffer,signame);        
	if (len>1) 
	{ 
          /* Vector signal, e.g. AD32 */
	  strcat(tc_err_result_buffer,"[");        
  	  sprintf(buf,"%lu",bitoffset);
	  strcat(tc_err_result_buffer,buf);        
	  strcat(tc_err_result_buffer,"]");
        }
	strcat(tc_err_result_buffer," violated\n");
      }
    }
  }

  if (accu4)
  {
    /* TC-violation somewhere in accu1 */
    for (i=0;i<32;i++)
    {
      /* check every bitposition */
      if (accu4&1<<i)
      {
        /* TC violation at bitposition i+96 */
	violation=1;
        err=BestTimCheckSignalGet(handle,i+96,NULL,&signame,&bitoffset,&len,NULL);
	B_ERRCHECK(err);
	strcat(tc_err_result_buffer,signame);        
	if (len>1) 
	{ 
          /* Vector signal, e.g. AD32 */
	  strcat(tc_err_result_buffer,"[");        
  	  sprintf(buf,"%lu",bitoffset);
	  strcat(tc_err_result_buffer,buf);        
	  strcat(tc_err_result_buffer,"]");
        }
	strcat(tc_err_result_buffer," violated\n");
      }
    }
  }

  if (violation==0)
  {
    /* no violation occured */
    strcat(tc_err_result_buffer,"************** NONE ***************\n");
  }

  *errortext=tc_err_result_buffer;
  
  B_ERRETURN(B_E_OK);
}

/* This functions checks, wether a timing violation has occured 
   at bitposition 
 */

b_errtype EXPORT BestTimCheckErrGet(
  b_handletype handle,
  b_int32 bitpos,
  b_int32 *value
)
{
  b_tcstatustype accureg;
  b_int32 status;

  B_DECLARE_FUNCNAME("BestTimCheckErrGet[tcerrget]");
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  if (!Best16BitRegisterFile(handle)) B_ERRETURN(B_E_NOT_E2925A);
  
  if (bitpos<32)
  {
    accureg=B_TC_ACCUSIG1;
  }
  else if (bitpos<64)
  {
    accureg=B_TC_ACCUSIG2;
  }
  else if (bitpos<96)
  {
    accureg=B_TC_ACCUSIG3;
  }  
  else
  {
    accureg=B_TC_ACCUSIG4;
  }

  bitpos %= 32;

  /* Now read status from card */
  err=BestTimCheckStatusGet(handle,accureg,&status);
  B_ERRCHECK(err);
  
  if (value) *value=(status&1<<bitpos?1:0);
  
  B_ERRETURN(B_E_OK);
}
